#!/usr/bin/perl -w

require 5.004;
use strict;
use File::Basename;

$ENV{PATH} = '/usr/local/bin:/usr/bin:/bin:/usr/local/sbin:/usr/sbin:/sbin';

if ($>) {
	print "You must be root to install SSHblock.\n";
	exit 1;
}

my %config;

$config{bindir} = ask("Where should I install the SSHblock binaries?", '/usr/local/sbin');
$config{history_file} = ask("Where should I save the history file?", '/etc/ssh/block-history');
if (-d $config{history_file}) {
	$config{history_file} =~ s/\/$//;
	print "$config{history_file} is a directory; expanding to $config{history_file}/block-history\n";
	$config{history_file} = "$config{history_file}/block-history";
}
$config{swatch_rule_file} = ask("Where should I install the swatch rules?", '/etc/swatch/sshblock');
if (-d $config{swatch_rule_file}) {
	$config{swatch_rule_file} =~ s/\/$//;
	print "$config{swatch_rule_file} is a directory; expanding to $config{swatch_rule_file}/swatch-rules\n";
	$config{swatch_rule_file} = "$config{swatch_rule_file}/swatch-rules";
}

$config{ip_whitelist} = ();
if (ask("Do you want to whitelist any IP addresses?", 'boolean', 0)) {
	print "Enter IP addresses, 1 per line. Enter an empty line to end.\n";
	while (<>) {
		chomp;
		last unless ($_);
		if (/^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/) {
			push @{$config{ip_whitelist}}, $_;
		} else {
			print "Not a valid IP address! Just hit <Enter> to stop entering IP addresses.\n";
		}
	}
} else {
	$config{ip_whitelist} = (' ');
}

chomp($config{hostname} = `hostname -f`);
$config{email_cutoff} = ask("Send email if address blocked this many times or more (0 to disable):", 2);
if (! $config{email_cutoff}) {
	$config{email_address} = '';
} else {
	$config{email_address} = ask("Send email to address:", 'root@' . `hostname -f`);
	my $email_host = (split /\@/, $config{email_address})[-1];
	if ($config{hostname} ne $email_host) {
		if (ask("When I send emails about blocked IP addresses, I plan to claim that\nthis machine's hostname is: $config{hostname}.\nShould I change that claim to $email_host instead?", 'boolean', 1)) {
			$config{hostname} = $email_host;
		}
	}
}


my %reqs = (
	'swatch' => "SSHblock requires swatch in order to work. Please download swatch\n(you can get it from http://sourceforge.net/projects/swatch/),\ninstall it, and then try running this installation script again.",
	'iptables' => "SSHblock requires iptables in order to work. If you don't have\niptables installed, you're probably not running Linux at all, so\nI'm not sure that there's anything you can do to install SSHblock.\nI'm terribly sorry.",
);

print "\n";

foreach my $req (keys %reqs) {
	print "Checking for $req... ";
	if (is_installed($req)) {
		print "ok\n";
	} else {
		print "not found!\n$reqs{$req}\n";
		exit 1;
	}
}

$config{std_cron_hourly} = 1;
if (! -d '/etc/cron.hourly') {
	$config{std_cron_hourly} = 0;
	print "You have no /etc/cron.hourly directory. SSHblock needs to run the\nsshunblock.pl script once an hour. I'd like to add the following\nentry to root's crontab:\n\n0 * * * *  $config{bindir}/sshunblock.pl\n\n";
	if (ask("Is that okay?", 'boolean', 1)) {
		my $cron = `crontab -l`;
		if (! $cron) {
			print "Couldn't read current crontab! You'll have to edit it yourself.\nPlease run \"crontab -e\", add:\n\n0 * * * *  $config{bindir}/sshunblock.pl\n\nto the crontab, and then run this install script again.\n";
			exit 1;
		}
		open CRON, "|crontab -" || die "Couldn't write to crontab: $!";
		print CRON "$cron\n";
		print CRON "0 * * * *    $config{bindir}/sshunblock.pl\n";
		if (! close CRON) {
			print "Couldn't write new crontab: $!";
			exit 1;
		}
	}
}

chomp($config{sendmail_path} = `which sendmail`);
if (! $config{sendmail_path}) {
	print "Couldn't find your system's sendmail binary or sendmail hook! Please enter\nthe path to an executable that can take a complete email message on STDIN,\n";
	$config{sendmail_path} = ask("like 'sendmail -t' (or blank to suppress email notices):", '');
	if (! $config{sendmail_path}) {
		$config{email_cutoff} = 0;
	}
}

$config{log_file} = find_log_file();
exit 1 unless ($config{log_file});

$config{rc_path} = find_rc_path();
exit 1 if ($config{rc_path} =~ /^a/i);
if ($config{rc_path} =~ /^\//) {
	my %stats = dir_stats($config{rc_path});
	if ($stats{rc_dot_names} / $stats{items} >= 0.75) {
		$config{rc_script_name} = 'rc.sshblock';
	} elsif ($stats{reg_names} / $stats{items} >= 0.75) {
		$config{rc_script_name} = 'sshblock';
	} else {
		$config{rc_script_name} = 'rc.sshblock.sh';
	}

	$config{runlevel} = (split /:/, `grep initdefault /etc/inittab`)[1];
	my $rc_parent = $config{rc_path};
	$rc_parent =~ s/\/[\.\w]+$//;
	my @poss_symlink_dirs = (
		"$config{rc_path}/rc$config{runlevel}.d",
		"$config{rc_path}/rc$config{runlevel}",
		"$rc_parent/rc.$config{runlevel}.d",
		"$rc_parent/rc.$config{runlevel}",
	);
	
	$config{symlink_path} = find_symlink_path();
	exit 1 if ($config{symlink_path} =~ /^a/i);
	if ($config{symlink_path} =~ /^\//) {
		opendir DIR, $config{symlink_path};
		while (my $item = readdir DIR) {
			if ($item =~ /^S(\d+)ssh/) {
				$config{symlink_name} = 'S' . ($1 + 1) . "sshblock";
				closedir DIR;
				last;
			}
		}
		closedir DIR;
		$config{symlink_name} = 'S99sshblock';
	}
}


unless (-d $config{bindir}) {
	my $ret = `mkdir -p $config{bindir}`;
	if ($? >> 8) {
		die "Couldn't create $config{bindir}: $!";
	} else {
		print "Created new directory $config{bindir}\n";
	}
}
process_file('sshblock.pl.tpl', "$config{bindir}/sshblock.pl");
chmod 0755, "$config{bindir}/sshblock.pl";
process_file('sshunblock.pl.tpl', "$config{bindir}/sshunblock.pl");
chmod 0755, "$config{bindir}/sshunblock.pl";
print "Installed SSHblock executables in $config{bindir}\n";


my $history_file_dir = dirname($config{history_file});
unless (-d $history_file_dir) {
	my $ret = `mkdir -p $history_file_dir`;
	if ($? >> 8) {
		die "Couldn't create $history_file_dir: $!";
	} else {
		print "Created new directory $history_file_dir\n";
	}
}
`touch $config{history_file}`;
print "Created empty history file $config{history_file}\n";


my $swatch_rule_file_dir = dirname($config{swatch_rule_file});
unless (-d $swatch_rule_file_dir) {
	my $ret = `mkdir -p $swatch_rule_file_dir`;
	if ($? >> 8) {
		die "Couldn't create $swatch_rule_file_dir: $!";
	} else {
		print "Created new directory $swatch_rule_file_dir\n";
	}
}
process_file('sshblock.tpl', $config{swatch_rule_file});
print "Created swatch rules in $config{swatch_rule_file}\n";

if (substr($config{rc_path}, 0, 1) eq '/') {
	process_file('rc.sshblock.tpl', "$config{rc_path}/$config{rc_script_name}");
	chmod 0755, "$config{rc_path}/$config{rc_script_name}";
	print "Installed rc/init script in $config{rc_path}\n";
	if (substr($config{symlink_path}, 0, 1) eq '/') {
		symlink "$config{rc_path}/$config{rc_script_name}", "$config{symlink_path}/$config{symlink_name}";
		print "SSHblock will auto-start on entry into runlevel $config{runlevel}\n";
	}
}
if ($config{std_cron_hourly}) {
	symlink "$config{bindir}/sshunblock.pl", "/etc/cron.hourly/sshunblock";
}

if (ask("Start SSHblock now?", 'boolean', 1)) {
	if (substr($config{rc_path}, 0, 1) eq '/') {
		`$config{rc_path}/$config{rc_script_name} start`;
		print `$config{rc_path}/$config{rc_script_name} status`;
	} else {
		`swatch -c $config{swatch_rule_file} -t $config{log_file} &> /dev/null &`;
	}
} else {
	if (substr($config{rc_path}, 0, 1) eq '/') {
		print "When you want to start SSHblock, just run:\n$config{rc_path}/$config{rc_script_name} start\n";
	} else {
		print "When you want to start SSHblock, just run:\nswatch -c $config{swatch_rule_file} -t $config{log_file} &> /dev/null &\n";
	}
}



exit 0;


sub ask {
	my ($question, $default) = @_[0,1];
	chomp $question;
	chomp $default;
	my $display_default = $default;
	my $bool = 0;
	if ($default eq 'boolean') {
		$default = $_[2];
		$display_default = $default ? 'Y/n' : 'y/N';
		$bool = 1;
	}
	print "$question [$display_default] ";
	chomp(my $result = <STDIN>);
	if ($bool) {
		return length($result) ? ($result !~ /^n/i) : $default;
	}
	return length($result) ? $result : $default;
}

sub dir_stats {
	my $dir = shift;
	$dir =~ s/\/$//;
	my %stats = (
		items => 0,
		files => 0,
		dirs => 0,
		links => 0,
		executables => 0,
		reg_names => 0,
		rc_dot_names => 0,
	);
	opendir DIR, $dir;
	while (my $item = readdir DIR) {
		$stats{items}++;
		$stats{dirs}++ if (-d "$dir/$item");
		$stats{links}++ if (-l "$dir/$item");
		if (-f "$dir/$item") {
			$stats{files}++;
			$stats{executables}++ if (-x "$dir/$item");
			$stats{reg_names}++ if ($item =~ /^[a-z]+$/i);
			$stats{rc_dot_names}++ if ($item =~ /^rc\./);
		}
	}
	closedir DIR;
	return %stats;
}

sub find_log_file {
	open SLC, "/etc/syslog.conf";
	while (my $line = <SLC>) {
		chomp($line);
		if ($line =~ /\*\.=?info/) {
			while (substr($line, -1) eq '\\') {
				$line .= <SLC>;
				chomp($line);
			}
			if ($line =~ /((\/\w+)+)$/) {
				close SLC;
				return $1;
			}
		}
	}
	close SLC;
	if (substr(`egrep "\\<sshd\\>" /var/log/messages | wc -l`, 0, -1)) {
		return '/var/log/messages';
	}
	if (substr(`egrep "\\<sshd\\>" /var/log/syslog | wc -l`, 0, -1)) {
		return '/var/log/syslog';
	}
	print "Cannot determine where your sshd logging output goes. (It doesn't\nseem to be in /var/log/messages or /var/log/syslog.)\n";
	return ask("Where do I find sshd's log output? (Enter to abort installation.)", '');
}

sub find_rc_path {
	my @poss_rc_dirs = qw(/etc/init.d /etc/rc.d /etc/rc.d/init.d);
	foreach my $dir (@poss_rc_dirs) {
		next unless (-d $dir);
		my %stats = dir_stats($dir);
		next unless ($stats{files} > 10);
		if ($stats{executables} / $stats{files} >= 0.5) {
			return $dir;
		}
	}
	print "Cannot determine where your rc scripts live. I can't find them in any of:\n* " . join("\n* ", @poss_rc_dirs) . "\n";
	print "Where do rc/init scripts live on your machine?\n";
	my $action = '';
	do {
		$action = ask("(Enter full pathname, 'a' to abort install, or 's' to skip\ninstalling rc script.)", 'a');
	} until ($action =~ /^[\/as]/i);
	return $action;
}

sub find_symlink_path {
	my @poss_symlink_dirs = ("/etc/init.d/rc$config{runlevel}.d", "/etc/rc.d/rc$config{runlevel}.d", "/etc/rc$config{runlevel}.d");
	foreach my $dir (@poss_symlink_dirs) {
		next unless (-d $dir);
		my %stats = dir_stats($dir);
#print "Got stats from dir $dir\n";
		next unless ($stats{files} > 10);
		if ($stats{symlinks} / $stats{files} >= 0.5) {
			return $dir;
		}
	}
	print "Can't determine where your startup symlinks live. I can't find them in any of:\n* " . join("\n* ", @poss_symlink_dirs) . "\n";
	print "Where do init symlinks (the S## and K## symlinks that point to\nactual init scripts) live on your machine?\n";
	my $action = '';
	do {
		$action = ask("(Enter full pathname, 'a' to abort install, or 's' to skip\ncreating startup symlink.)", 'a');
	} until ($action =~ /^[\/as]/i);
	return $action;
}

sub is_installed {
	my $prog = shift;
	`which $prog 2>/dev/null | grep $prog >/dev/null`;
	return 1 - ($? >> 8);
}

sub process_file {
	my $src = shift;
	my $dest = shift;
	if (! $dest) {
		$dest = $src;
		$dest =~ s/\.tpl$//;
	}
	open IN, $src || die "Couldn't open $src: $!";
	open OUT, ">$dest" || die "Couldn't open $dest: $!";
	while (<IN>) {
		if (/\[\[(\w+)\]\]/) {
			my $lvar = lc($1);
			my $repl = $config{$lvar};
			no strict 'refs';
			if ($#{$config{$lvar}} > -1) {
				$repl = join(" ", @{$config{$lvar}});
			}
			use strict 'refs';
			s/\[\[(\w+)\]\]/$repl/g;
		}
		print OUT $_;
	}
	close IN;
	my $ret = close OUT || die "Couldn't write $dest: $!";
}


